/* ======================================================== */
/* global require */
const gulp = require( 'gulp' ); // Собственно gulp
const pump = require( 'pump' ); // отслеживание ошибок

const concat = require( 'gulp-concat' ); // объединение набора файлов в один
const include = require( 'gulp-file-include' ); // включение файлов
const clean = require( 'gulp-clean' ); // удаление файлов и каталогов

const postcss = require( 'gulp-postcss' ); // Пост-обработка CSS
const postcssPresetEnv = require( 'postcss-preset-env' ); // CSS Next
const cssvariables = require('postcss-css-variables'); // полифил для CSS-переменных
const atImport = require('postcss-import'); // парсинг и включение импорта CSS
const sourcemaps = require( 'gulp-sourcemaps' ); // карты исходного кода

const webpack = require( 'webpack-stream' );
const compiler = require( 'webpack' );

const imagemin = require( 'gulp-image' );  // оптимизация картинок
const cache = require( 'gulp-cache' );

const browserSync = require( 'browser-sync' ).create(); // обновление налету

/* ========================================================== */

/**
 * Библиотеки CSS
 */
const libCSS = [
	// 'src/assets/fonts/stylesheet.css',
	// 'node_modules/bootstrap/dist/css/bootstrap.css'
];

/**
 * Настройки CSS
 */
const postcssOptions = [
	atImport(),
	postcssPresetEnv({
		features: {
			'custom-properties': false
		},
		stage: 0,
		autoprefixer: {
			grid: true,
			browsers: [ 'last 4 versions', 'ie >= 11', 'iOS >= 9' ]
		}
	}),
	cssvariables()
];

/**
 * Сделать один CSS
 */
const isOneCSS = true;

/**
 * Сделать один JS
 */
const isOneJS = true;

/* ================================================================= */

/**
 * CLEAN
 * Полная очистка
 */
gulp.task( 'CLEAN', function( cb ){
	pump([
		gulp.src( [ './dev', './prod' ], { read: false } ),
		clean()
	], cb );
});


/**
 * CSS
 */
gulp.task( 'dev:css', function( cb ){

	// Собираем вендорные либы
	pump([
		gulp.src( libCSS ),
		concat( 'vendor.css' ),
		gulp.dest( './dev/assets/css' )
	]);

	// Собираем свои стили
	pump([
		gulp.src( './src/assets/css/*.css' ),
		sourcemaps.init(),
		postcss( postcssOptions ),
		sourcemaps.write( '.' ),
		gulp.dest( './dev/assets/css' ),
		browserSync.stream()
	], cb );
});


// Собираем CSS
gulp.task( 'prod:css', function( cb ){
	// Собираем вендорные либы
	pump([
		gulp.src( libCSS ),
		concat( 'vendor.css' ),
		gulp.dest( './prod/assets/css' )
	]);

	// Собираем свои стили
	pump([
		gulp.src( './src/assets/css/*.css' ),
		sourcemaps.init(),
		postcss( postcssOptions ),
		sourcemaps.write( '.' ),
		gulp.dest( './prod/assets/css' ),
	], cb );
});

// ================== JS ============================
// Компилируем и собираем все js файлы проекта
gulp.task( 'dev:js', function( cb ){

	// ES5
	pump([
		gulp.src( './src/assets/js/*.js' ),
		webpack( require( './webpack.dev.js' ), compiler, function( err, stats ){
			// console.log( err, stats );
		}),
		gulp.dest( './dev/assets/js' )
	], cb );

});


// Обрабатываем js файлы библиотек и проекта
gulp.task( 'prod:js', function(){

	// ES5
	pump([
		gulp.src( './src/assets/js/*.js' ),
		webpack( require( './webpack.prod.js' ), compiler, function( err, stats ){
			// console.log( err, stats );
		}),
		gulp.dest( './prod/assets/js' )
	]);

});


// ================== IMG ============================
// Оптимизация картинок для dev
gulp.task('dev:img', function( cb ) {
	pump([
		gulp.src( [
			'./src/assets/img/**/*',
			'./src/components/**/*.png',
			'./src/components/**/*.jpg',
			'./src/components/**/*.jpeg',
			'./src/components/**/*.gif',
			'./src/components/**/*.svg' ]
		),
		cache(
			imagemin()
		),
		gulp.dest( './dev/assets/img' )
	], cb );
});

// Оптимизация картинок для prod
gulp.task('prod:img', function( cb ) {
	pump([
		gulp.src( [
			'./src/assets/img/**/*',
			'./src/components/**/*.png',
			'./src/components/**/*.jpg',
			'./src/components/**/*.jpeg',
			'./src/components/**/*.gif',
			'./src/components/**/*.svg' ]
		),
		cache(
			imagemin()
		),
		gulp.dest( './prod/assets/img' )
	], cb );
});


// ================== HTML ============================
// HTML для dev
gulp.task( 'dev:html', function( cb ){

	// HTML
	pump([
		gulp.src( [
			'./src/**/*.html',
			'!./src/assets/*.html',
			'!./src/components/**/*.html']
		),
		include({
			prefix: '@@',
			basepath: '@file',
			context: {
				rev: +new Date
			}
		}),
		gulp.dest( './dev' )
	], cb );

	pump([
		gulp.src( [
			'./src/favicon.ico',
			'./src/robots.txt'
		] ),
		gulp.dest( './dev' )
	]);
});


// HTML для prod
gulp.task( 'prod:html', function( cb ){
	pump([
		gulp.src([
			'./src/**/*.html',
			'!./src/assets/*.html',
			'!./src/components/**/*.html'
		]),
		include({
			prefix: '@@',
			basepath: '@file',
			context: {
				rev: +new Date
			}
		}),
		gulp.dest( './prod' )
	], cb );

	pump([
		gulp.src( [
			'./src/favicon.ico',
			'./src/robots.txt',
		] ),
		gulp.dest( './prod' )
	]);

});


// ================== DEV ============================
// Слежение за dev
gulp.task( 'DEV', [ 'dev:img', 'dev:css', 'dev:js', 'dev:html' ], function() {

	browserSync.init({
		server: {
			baseDir: './dev/'
		}
	});

	gulp.watch( './src/assets/css/*.css' , ['dev:css']);
	gulp.watch( './src/components/**/*.css' , ['dev:css']);

	gulp.watch( './src/assets/js/*.js', [ 'dev:js' ] ).on('change', browserSync.reload);
	gulp.watch( './src/components/**/*.js', [ 'dev:js' ] ).on('change', browserSync.reload);
	gulp.watch( './dev/assets/js/**/*' ).on('change', browserSync.reload);

	gulp.watch( './src/**/*.html', [ 'dev:html' ] );
	gulp.watch( './src/components/**/*.html', [ 'dev:html' ] );
	gulp.watch( './dev/*.html' ).on('change', browserSync.reload);

	gulp.watch( './src/assets/img/**/*', [ 'dev:img' ] );
	gulp.watch( './src/components/**/*.gif', [ 'dev:img' ] );
	gulp.watch( './src/components/**/*.jpg', [ 'dev:img' ] );
	gulp.watch( './src/components/**/*.jpeg', [ 'dev:img' ] );
	gulp.watch( './src/components/**/*.png', [ 'dev:img' ] );
	gulp.watch( './src/components/**/*.svg', [ 'dev:img' ] );
	gulp.watch( './dev/assets/img/**/*' ).on( 'change', browserSync.reload );
});

// ================== PROD ============================
// Публикация на prod
gulp.task( 'PROD', function(){
	gulp.start( 'prod:img', 'prod:css', 'prod:js', 'prod:html' );
});